#!/usr/bin/bash
set -eo pipefail
usage_exit() {
  echo "usage: $0 <image_file_path> <src_file_path> <result_directory> <hw> <task>"
  echo "example: $0 /path/to/the_docker_image test.src results CPU-ALL throughput"
  exit 1
}

if [ $# -lt 3 ]; then usage_exit; fi


image_file_path=$1; shift; echo "inspector: info: image_file_path: ${image_file_path}"
src_file_path=$1; shift; echo "inspector: info: src_file_path: ${src_file_path}"
result_directory=$1; shift; echo "inspector: info: result_directory: ${result_directory}"
opt_hw=$1; shift; echo "inspector: info: hardware: ${opt_hw}"
opt_task=$1; shift; echo "inspector: info: task: ${opt_task}"

rm -rf $result_directory 2>/dev/null || true
mkdir -p $result_directory

opt_cpu=""
opt_nvidia=""
opt_memory="--memory=500g"

case $opt_hw in
  CPU-ALL)
    opt_priv="--privileged"
    ;;
  CPU-1)
    opt_cpu="--cpuset-cpus=0"
    ;;
  GPU)
    opt_nvidia="--runtime=nvidia --gpus 0"
    ;;
  *)
    echo $opt_hw is not a valid hardware setup 1>&2
    exit 1
esac
case $opt_task in
  latency)
    if [ $opt_hw == "CPU-ALL" ]; then
      echo $opt_hw $opt_task is not a task 1>&2
      exit 1
    fi
    ;;
  throughput)
    ;;
  *)
    echo $opt_task is not a valid task: choose latency or throughput 1>&2
    exit 1
    ;;
esac
     
echo "inspector: hardware platform: ${opt_hw}"
echo "inspector: task: ${opt_task}"
echo "inspector: info: opt_cpu: ${opt_cpu}"
echo "inspector: info: opt_memory: ${opt_memory}"

# if [ -e ${result_directory} ]; then
#   echo "Directory '${result_directory}' already exists."
#   exit 1
# fi
mkdir -p ${result_directory}

echo "inspector: proc: Obtaining image size ..."
image_file_size=$(ls -l ${image_file_path} | cut -d ' ' -f 5)
echo ${image_file_size} > ${result_directory}/image_file_size
echo "inspector: info: image_file_size: ${image_file_size}"

# clean up
docker kill $(docker ps -q) 2>/dev/null || true

echo "inspector: proc: Loading image ..."
image_name=$(docker load -i ${image_file_path} |cut -d " " -f 3)

echo "info: image_name: ${image_name}"

if [ $opt_hw == GPU ]; then
  echo "inspector: proc: Launching nvidia-smi ..."
  nvidia-smi --query-gpu=memory.used --format=csv,noheader,nounits -l 1 -f ${result_directory}/gpumem_usage_history &
  nvidia_smi_pid=$!
fi

echo "nvidia gpu options: ${opt_nvidia}"
echo "inspector: proc: Launching container ..."
echo "docker run ${opt_priv} -itd ${opt_cpu} ${opt_memory} --memory-swap=0 ${opt_nvidia} ${image_name} /bin/sh"
container_id=$(docker run ${opt_priv} -itd ${opt_cpu} ${opt_memory} --memory-swap=0 ${opt_nvidia} ${image_name} /bin/sh)

echo "container_id: ${container_id}"

docker cp -a ${container_id}:/model ${result_directory}/model

#Hotfix for huaweitsc
if [ $(basename ${image_file_path} |cut -d _ -f 1) == huaweitsc ]; then
  docker cp libbolt.so ${container_id}:/home/tsc/libs/libbolt.so
fi

echo "waiting for the container"
sleep 20

#Hotfix for uedin
#if [ $(basename ${image_file_path}) == uedin_student.02.12-1.tiny.pruned.heads.efh_0.3.image ] || [ $(basename ${image_file_path}) == uedin_student.02.12-2.tied.tiny.pruned.heads.efh_0.3.image ]; then
#  docker exec ${container_id} ln -s /extracted-model/model.npz.* /extracted-model/model.npz
#fi

if [ ${opt_task} == "throughput" ]; then
  echo "inspector: proc: Running throughput test..."
  (time docker exec -i "${container_id}" /run.sh ${opt_hw} throughput <${src_file_path} >${result_directory}/hyp) >${result_directory}/run.stdout 2>${result_directory}/run.stderr
else
  echo "inspector: proc: Running latency test..."
  docker cp latency ${container_id}:/
  (time docker exec -i "${container_id}" /latency /run.sh ${opt_hw} latency <${src_file_path} >${result_directory}/hyp) >${result_directory}/run.stdout 2>${result_directory}/run.stderr
fi

echo "inspector: proc: Obtaining CPU usage ..."
cpu_usage=$(cat /sys/fs/cgroup/cpuacct/docker/${container_id}/cpuacct.usage)
echo ${cpu_usage} > ${result_directory}/cpu_usage
echo "inspector: info: cpu_usage(ns): ${cpu_usage}"

echo "inspector: proc: Obtaining memory usage ..."
max_memory_usage=$(cat /sys/fs/cgroup/memory/docker/${container_id}/memory.max_usage_in_bytes)
echo ${max_memory_usage} > ${result_directory}/max_memory_usage
echo "inspector: info: max_memory_usage(byte): ${max_memory_usage}"

if [ $opt_hw == GPU ]; then
  echo "inspector: proc: Terminating nvidia-smi ..."
  kill -SIGINT ${nvidia_smi_pid}
  kill_status=$?
  while [ ${kill_status} -eq 0 ]; do
    kill_status=0
    kill -0 ${nvidia_smi_pid} 2> /dev/null || kill_status=$?
  done

  echo "inspector: proc: Obtaining GPU memory usage ..."
  max_gpumem_usage=$(awk 'BEGIN { m = 0; } { if ($1 > m) m = $1; } END { print m; }' ${result_directory}/gpumem_usage_history)
  echo ${max_gpumem_usage} > ${result_directory}/max_gpumem_usage
  echo "inspector: info: max_gpumem_usage(MB): ${max_gpumem_usage}"
fi

echo "inspector: proc: Removing Docker image and container ..."
docker stop ${container_id} > /dev/null
docker rm -f ${container_id} > /dev/null
docker rmi -f ${image_name} > /dev/null
