#!/usr/bin/env python
# coding=utf-8
# Copyright 2021 The HuggingFace Team All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Fine-tuning the library's seq2seq models for question answering using the 🤗 Seq2SeqTrainer.
"""
# You can also adapt this script on your own question answering task. Pointers for this are left as comments.
import logging
import os
import sys
import random
from dataclasses import dataclass, field
from typing import List, Optional, Tuple
from tqdm import tqdm
from collections import Counter

import torch
import datasets
from datasets import load_dataset, DatasetDict, concatenate_datasets, load_from_disk
import numpy as np
import transformers
from transformers import (
    AutoConfig,
    AutoModelForSeq2SeqLM,
    AutoTokenizer,
    DataCollatorForSeq2Seq,
    HfArgumentParser,
    set_seed
)
from transformers.trainer_utils import get_last_checkpoint
from transformers.utils import check_min_version

from src.utils import ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments, \
                        TASK_DEMONSTRATION_ONLY, get_dataset_sizes
from src.metrics import Bleu, handle_metrics
from src.models import HyperLoRAConfig, HyperLoRAModelForPretrain
from src.trainer import Seq2SeqHyperTrainer
from src.dataset.glue_tasks import AutoTask

import os
os.environ["WANDB_PROJECT"]="HyperLoRA"

logger = logging.getLogger(__name__)

def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, Seq2SeqArguments, LoraArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args, lora_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args, lora_args = parser.parse_args_into_dataclasses()

    # update model_args

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # training_args.local_rank = -1
    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f", distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # read multi-task datasets
    train_dataset_sizes, val_dataset_sizes, test_dataset_sizes = [], [], []
    raw_train, raw_validation, raw_test, task2demonstrations = [], [], [], {}
    all_task_files = os.listdir(data_args.dataset_name)
    print(all_task_files)
    for task_file in tqdm(all_task_files, desc='Loading Datasets', total=len(all_task_files)):
        train_dataset = AutoTask.get(task_file, seed=training_args.seed).get_dataset(split="train",
                        n_obs=-1, add_prefix=True)

        # TODO: sample n_demonstrations from train_task_dataset
        # rnd_demo_idx = random.sample(range(len(train_dataset)), data_args.n_demonstrations)
        # demo_datasets = train_dataset.select(rnd_demo_idx)
        # filter sampled demostrations
        # train_dataset = train_dataset.filter(lambda x: x not in rnd_demo_idx).shuffle()
        # TODO: Auto Demo
        demo_datasets = load_dataset("json", data_files=os.path.join(f'data/glue_auto_demonstration', f'{task_file}_train.json'))['train']
        task2demonstrations[task_file] = demo_datasets

        # val
        eval_dataset = AutoTask.get(task_file, seed=training_args.seed).get_dataset(split="validation",
                            n_obs=-1, add_prefix=True, split_validation_test=True)
        # test
        test_dataset = AutoTask.get(task_file, seed=training_args.seed).get_dataset(split="test",
                            n_obs=-1, add_prefix=True, split_validation_test=True)

        train_dataset_sizes.append(len(train_dataset))
        val_dataset_sizes.append(len(eval_dataset))
        test_dataset_sizes.append(len(test_dataset))

        raw_train.append(train_dataset)
        raw_validation.append(eval_dataset)
        raw_test.append(test_dataset)

    raw_datasets = DatasetDict({"train": concatenate_datasets(raw_train),
                                "validation": concatenate_datasets(raw_validation),
                                "test": concatenate_datasets(raw_test)})
    demo_datasets = DatasetDict(task2demonstrations)


    # Load pretrained model and tokenizer
    #
    # Distributed training:
    # The .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    hyperlora_config = HyperLoRAConfig.from_pretrained(
        model_args.hypelora_name_or_path, lora_rank=lora_args.lora_rank
    )
    plm_config = AutoConfig.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    plm_model = AutoModelForSeq2SeqLM.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=plm_config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
        torch_dtype=torch.bfloat16
    )

    # We resize the embeddings only when necessary to avoid index errors. If you are creating a model from scratch
    # on a small vocab and want a smaller embedding size, remove this test.
    embedding_size = plm_model.get_input_embeddings().weight.shape[0]
    if len(tokenizer) > embedding_size:
        plm_model.resize_token_embeddings(len(tokenizer))

    if plm_model.config.decoder_start_token_id is None:
        raise ValueError("Make sure that `config.decoder_start_token_id` is correctly defined")

    # task2id
    all_task_names = demo_datasets.keys()
    model = HyperLoRAModelForPretrain(config=hyperlora_config,
                                      model_args=model_args,
                                      lora_args=lora_args,
                                      encoder=plm_model.encoder,
                                      pretrain_task_names=all_task_names)
    task2id = model.task2id

    # Preprocessing the datasets.
    # We need to generate and tokenize inputs and targets.
    if training_args.do_train:
        column_names = raw_datasets["train"].column_names
    elif training_args.do_eval:
        column_names = raw_datasets["validation"].column_names
    else:
        logger.info("There is nothing to do. Please pass `do_train`, `do_eval` and/or `do_predict`.")
        return
    # column_names = ["input", "output"]
    # Get the column names for input/target.
    input_column = data_args.input_column
    output_column = data_args.output_column

    # Temporarily set max_answer_length for training.
    max_answer_length = data_args.max_answer_length
    padding = "max_length" if data_args.pad_to_max_length else False

    if training_args.label_smoothing_factor > 0 and not hasattr(model, "prepare_decoder_input_ids_from_labels"):
        logger.warning(
            "label_smoothing is enabled but the `prepare_decoder_input_ids_from_labels` method is not defined for"
            f"`{model.__class__.__name__}`. This will lead to loss being calculated twice and will take up more memory"
        )

    max_seq_length = data_args.max_seq_length
    max_demo_length = data_args.max_demo_length

    def preprocess_demonstrations(task_demo_examples):
        demonstration_inputs = ""
        inputs = task_demo_examples[input_column]
        targets = task_demo_examples[output_column]
        assert len(inputs) == data_args.n_demonstrations
        for i in range(len(inputs)):
            demonstration_inputs += TASK_DEMONSTRATION_ONLY.format(input=inputs[i], target=targets[i])

        demo_tokenize_output= tokenizer(demonstration_inputs.strip(), max_length=max_demo_length, padding=padding, truncation=True)
        demo_output = {'demo_input_ids': demo_tokenize_output['input_ids'],
                       'demo_attention_mask': demo_tokenize_output['attention_mask']
                       }
        return demo_output

    def preprocess_function(examples):
        inputs = examples[input_column]
        targets = examples[output_column]
        task_names = examples['task']

        if data_args.is_dynamic_demo:
            task_ids, demo_input_ids, demo_attention_mask = [], [], []
            for task in task_names:
                rnd_demo_idx = random.sample(range(len(demo_datasets[task])), data_args.n_demonstrations)
                rnd_demos = demo_datasets[task].select(rnd_demo_idx)
                demo_input_texts, demo_targets = rnd_demos[input_column], rnd_demos[output_column]
                demonstration_inputs = ''
                for i in range(len(demo_input_texts)):
                    demonstration_inputs += TASK_DEMONSTRATION_ONLY.format(input=demo_input_texts[i], target=demo_targets[i])
                demo_tokenize_output= tokenizer(demonstration_inputs.strip(), max_length=max_seq_length, padding='max_length', truncation=True)
                task_ids.append(task2id[task])
                demo_input_ids.append(demo_tokenize_output['input_ids'])
                demo_attention_mask.append(demo_tokenize_output['attention_mask'])
        else:
            task_ids, demo_input_ids, demo_attention_mask = [], [], []
            for task in task_names:
                task_ids.append(task2id[task])
                demo_input_ids.append(demo_datasets[task]['demo_input_ids'])
                demo_attention_mask.append(demo_datasets[task]['demo_attention_mask'])

        model_inputs = tokenizer(inputs, max_length=max_seq_length, padding=padding, truncation=True)
        # Tokenize targets with text_target=...
        labels = tokenizer(text_target=targets, max_length=max_answer_length, padding=padding, truncation=True)

        # padding in the loss.
        if padding == "max_length" and data_args.ignore_pad_token_for_loss:
            labels["input_ids"] = [
                [(l if l != tokenizer.pad_token_id else -100) for l in label] for label in labels["input_ids"]
            ]

        model_inputs["labels"] = labels["input_ids"]
        model_inputs["task_ids"] = task_ids
        model_inputs["demo_input_ids"] = demo_input_ids
        model_inputs["demo_attention_mask"] = demo_attention_mask
        return model_inputs

    # prepreocess demonstration
    if not data_args.is_dynamic_demo:
        for key in demo_datasets:
            demo_datasets[key] = preprocess_demonstrations(demo_datasets[key])

    if training_args.do_train:
        if "train" not in raw_datasets:
            raise ValueError("--do_train requires a train dataset")
        train_dataset = raw_datasets["train"]
        if data_args.max_train_samples is not None:
            # We will select sample from whole data if agument is specified
            max_train_samples = min(len(train_dataset), data_args.max_train_samples)
            train_dataset = train_dataset.select(range(max_train_samples))
        # Create train feature from dataset
        with training_args.main_process_first(desc="train dataset map pre-processing"):
            train_dataset = train_dataset.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on train dataset",
            )
        if data_args.max_train_samples is not None:
            # Number of samples might increase during Feature Creation, We select only specified max samples
            max_train_samples = min(len(train_dataset), data_args.max_train_samples)
            train_dataset = train_dataset.select(range(max_train_samples))

    if training_args.do_eval:
        if "validation" not in raw_datasets:
            raise ValueError("--do_eval requires a validation dataset")
        eval_examples = raw_datasets["validation"]
        if data_args.max_eval_samples is not None:
            # We will select sample from whole data
            max_eval_samples = min(len(eval_examples), data_args.max_eval_samples)
            eval_examples = eval_examples.select(range(max_eval_samples))
        # Validation Feature Creation
        with training_args.main_process_first(desc="validation dataset map pre-processing"):
            eval_dataset = eval_examples.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on validation dataset",
            )
        if data_args.max_eval_samples is not None:
            # During Feature creation dataset samples might increase, we will select required samples again
            max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
            eval_dataset = eval_dataset.select(range(max_eval_samples))

    if training_args.do_test:
        if "test" not in raw_datasets:
            raise ValueError("--do_eval requires a validation dataset")
        test_examples = raw_datasets["test"]
        if data_args.max_test_samples is not None:
            # We will select sample from whole data
            max_test_samples = min(len(test_examples), data_args.max_test_samples)
            test_examples = test_examples.select(range(max_test_samples))
        # Validation Feature Creation
        with training_args.main_process_first(desc="test dataset map pre-processing"):
            test_dataset = test_examples.map(
                preprocess_function,
                batched=True,
                num_proc=data_args.preprocessing_num_workers,
                remove_columns=column_names,
                load_from_cache_file=not data_args.overwrite_cache,
                desc="Running tokenizer on test dataset",
            )
        if data_args.max_test_samples is not None:
            # During Feature creation dataset samples might increase, we will select required samples again
            max_test_samples = min(len(test_dataset), data_args.max_test_samples)
            test_dataset = test_dataset.select(range(max_test_samples))


    # Data collator
    label_pad_token_id = -100 if data_args.ignore_pad_token_for_loss else tokenizer.pad_token_id
    data_collator = DataCollatorForSeq2Seq(
        tokenizer,
        model=model,
        label_pad_token_id=label_pad_token_id,
        pad_to_multiple_of=8 if training_args.fp16 else None
    )

    bleu_eval = Bleu()

    def get_exact_match(predictions: List[str], references: List[str], **kwargs):
        """
        Exact match as the default evaluation
        """
        assert len(predictions) == len(references)
        correct_num = 0
        for prediction, reference in zip(predictions, references):
            if prediction.lower() == reference.lower():
                correct_num += 1
        return correct_num / len(predictions)

    def compute_metrics(eval_preds):
        preds, labels = eval_preds
        if isinstance(preds, tuple):
            preds = preds[0]

        if data_args.ignore_pad_token_for_loss:
            # Replace -100 in the labels as we can't decode them.
            labels = np.where(labels != -100, labels, tokenizer.pad_token_id)
            preds = np.where(preds != -100, preds, tokenizer.pad_token_id)

        decoded_preds = tokenizer.batch_decode(preds, skip_special_tokens=True)
        decoded_labels = tokenizer.batch_decode(labels, skip_special_tokens=True)

        # Some simple post-processing
        decoded_preds, decoded_labels = postprocess_text(decoded_preds, decoded_labels)
        result = get_exact_match(predictions=decoded_preds, references=decoded_labels)
        bleu_result = bleu_eval.compute(predictions=decoded_preds, references=decoded_labels)
        return {"exact_match": result, "bleu": bleu_result["bleu"]}

    # Post-processing:
    def postprocess_text(preds, labels):
        preds = [pred.strip() for pred in preds]
        labels = [label.strip() for label in labels]
        return preds, labels

    # no shuffle in inter-task
    # train_dataset = train_dataset.shuffle(seed=training_args.seed)
    # call_backs = [EarlyStoppingCallback(early_stopping_patience=3)]
    # Initialize our Trainer
    trainer = Seq2SeqHyperTrainer(
        train_dataset_sizes=train_dataset_sizes,
        val_dataset_sizes=val_dataset_sizes,
        test_dataset_sizes=test_dataset_sizes,
        model=model,
        args=training_args,
        train_dataset=train_dataset if training_args.do_train else None,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        tokenizer=tokenizer,
        data_collator=data_collator,
        compute_metrics=compute_metrics if training_args.predict_with_generate else None,
        # callbacks=call_backs
    )

    # Training
    if training_args.do_train:
        checkpoint = None
        if training_args.resume_from_checkpoint is not None:
            checkpoint = training_args.resume_from_checkpoint
        train_result = trainer.train(resume_from_checkpoint=checkpoint)
        trainer.save_model()  # Saves the tokenizer too for easy upload

        metrics = train_result.metrics
        max_train_samples = (
            data_args.max_train_samples if data_args.max_train_samples is not None else len(train_dataset)
        )
        metrics["train_samples"] = min(max_train_samples, len(train_dataset))

        trainer.log_metrics("train", metrics)
        trainer.save_metrics("train", metrics)
        trainer.save_state()

    # Evaluation
    max_length = (
        training_args.generation_max_length
        if training_args.generation_max_length is not None
        else data_args.val_max_answer_length
    )
    num_beams = data_args.num_beams if data_args.num_beams is not None else training_args.generation_num_beams
    if training_args.do_eval:
        logger.info("*** Evaluate ***")
        metrics = trainer.evaluate(max_length=max_length, num_beams=num_beams, metric_key_prefix="eval")

        max_eval_samples = data_args.max_eval_samples if data_args.max_eval_samples is not None else len(eval_dataset)
        metrics["eval_samples"] = min(max_eval_samples, len(eval_dataset))

        trainer.log_metrics("eval", metrics)
        trainer.save_metrics("eval", metrics)

    if training_args.do_test:
        logger.info("*** Test ***")
        trainer.eval_dataset = test_dataset
        trainer.val_dataset_sizes = test_dataset_sizes
        metrics = trainer.evaluate(test_dataset)
        if trainer.is_world_process_zero():
            handle_metrics(logger, "test", metrics, training_args.output_dir)


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()